/* FreeHTML Offline Photo Editor (WebGL)
   - Fit-to-view by default (fixes “image loads tiny”)
   - WebGL shader pipeline with: exposure, contrast, saturation, temperature, rotation
   - Curves: draggable points -> 256 LUT (RGB + per-channel)
   - HSL: 8 hue ranges with per-range Hue/Sat/Lum adjustments (approx)
   - Local masks: linear gradient + radial mask overlay as alpha map in shader
*/

(() => {
  "use strict";

  // ---------- DOM ----------
  // Quick controls (on-canvas)
const qcZoom = document.getElementById("qcZoom");
const qcRotate = document.getElementById("qcRotate");
const qcZoomOut = document.getElementById("qcZoomOut");
const qcZoomIn = document.getElementById("qcZoomIn");
const qcFit = document.getElementById("qcFit");

const qcRotL = document.getElementById("qcRotL");
const qcRotR = document.getElementById("qcRotR");
const qcRot0 = document.getElementById("qcRot0");

const quickControls = document.getElementById("quickControls");

  const glCanvas = document.getElementById("glCanvas");
  const maskCanvas = document.getElementById("maskCanvas");
  const curveCanvas = document.getElementById("curveCanvas");
  const viewerStage = document.getElementById("viewerStage");

  const fileInput = document.getElementById("fileInput");
  const btnOpen = document.getElementById("btnOpen");
  const btnOpen2 = document.getElementById("btnOpen2");
  const btnFit = document.getElementById("btnFit");
  const btnReset = document.getElementById("btnReset");
  const btnExport = document.getElementById("btnExport");
  const btnTogglePanel = document.getElementById("btnTogglePanel");
  const btnClosePanel = document.getElementById("btnClosePanel");
  const rightPanel = document.getElementById("rightPanel");

  const detectedType = document.getElementById("detectedType");
  const rawStatus = document.getElementById("rawStatus");

  const hudName = document.getElementById("hudName");
  const hudInfo = document.getElementById("hudInfo");
  const hudZoom = document.getElementById("hudZoom");
  const hudRotate = document.getElementById("hudRotate");
  const hudMode = document.getElementById("hudMode");

  const wheelMode = document.getElementById("wheelMode");
  const qualityMode = document.getElementById("qualityMode");

  // Basic sliders
  const exposure = bindSlider("exposure", "exposureVal", v => formatNum(v));
  const contrast = bindSlider("contrast", "contrastVal", v => formatNum(v));
  const saturation = bindSlider("saturation", "saturationVal", v => formatNum(v));
  const temperature = bindSlider("temperature", "temperatureVal", v => formatNum(v));
  const rotate = bindSlider("rotate", "rotateVal", v => `${Math.round(v)}°`);

  // Mask sliders (applied only where mask alpha > 0)
  const maskExposure = bindSlider("maskExposure", "maskExposureVal", v => formatNum(v));
  const maskSaturation = bindSlider("maskSaturation", "maskSaturationVal", v => formatNum(v));
  const btnClearMask = document.getElementById("btnClearMask");

  // Modes
  const modeViewerBtn = document.getElementById("modeViewer");
  const modeLinearBtn = document.getElementById("modeLinear");
  const modeRadialBtn = document.getElementById("modeRadial");
  const modeEraseBtn = document.getElementById("modeErase");

  // HSL tabs
  const hslHue = bindSlider("hslHue", "hslHueVal", v => formatNum(v));
  const hslSat = bindSlider("hslSat", "hslSatVal", v => formatNum(v));
  const hslLum = bindSlider("hslLum", "hslLumVal", v => formatNum(v));
  const hslTabs = Array.from(document.querySelectorAll(".tab"));

  // Curve tabs + buttons
  const curveTabs = Array.from(document.querySelectorAll(".cTab"));
  const btnCurveAdd = document.getElementById("btnCurveAdd");
  const btnCurveReset = document.getElementById("btnCurveReset");

  // ---------- Utilities ----------
  function formatNum(v){
    // user preference: avoid decimals where possible
    const n = Number(v);
    const rounded = Math.round(n * 100) / 100;
    if (Math.abs(rounded - Math.round(rounded)) < 0.0001) return String(Math.round(rounded));
    return String(rounded);
  }

  function bindSlider(id, outId, fmt){
    const el = document.getElementById(id);
    const out = document.getElementById(outId);
    const api = {
      el,
      get value(){ return Number(el.value); },
      set value(v){ el.value = String(v); out.textContent = fmt(v); },
      onChange(fn){
        el.addEventListener("input", () => { out.textContent = fmt(el.value); fn(Number(el.value)); });
      }
    };
    out.textContent = fmt(el.value);
    return api;
  }

  function clamp01(x){ return Math.max(0, Math.min(1, x)); }

  function isRawLike(name){
    const ext = (name.split(".").pop() || "").toLowerCase();
    return ["dng","nef","cr2","cr3","arw","raf","rw2","orf","srw","pef"].includes(ext);
  }

  // ---------- State ----------
  const state = {
    fileName: "",
    imgW: 0,
    imgH: 0,

    // Viewer transform
    zoom: 1,
    panX: 0, // in screen px
    panY: 0,
    rotateDeg: 0,

    // mode
    mode: "viewer", // viewer | linear | radial | erase

    // HSL per range (8 ranges)
    hslActive: 0, // index 0..7
    hsl: Array.from({length:8}, () => ({ h:0, s:0, l:0 })),

    // Curves points per channel
    activeCurve: "rgb", // rgb|r|g|b
    curves: {
      rgb: defaultCurvePoints(),
      r: defaultCurvePoints(),
      g: defaultCurvePoints(),
      b: defaultCurvePoints()
    },

    // mask
    maskEnabled: true,
    maskExposure: 0,
    maskSaturation: 0
  };

  function defaultCurvePoints(){
    // points in [0..1] space (x input, y output)
    return [
      {x:0, y:0},
      {x:1, y:1}
    ];
  }

  // ---------- WebGL ----------
  let gl, program;
  let texImage = null;    // image texture
  let texMask = null;     // mask alpha texture
  let texCurve = null;    // curve LUT texture (RGBA 256x1)
  let quadVBO = null;
  let u = {};             // uniforms
  let imgBitmap = null;   // decoded image (ImageBitmap)
  let cpuCanvas = null;   // for export readback

  const VERT = `
  attribute vec2 a_pos;
  varying vec2 v_uv;
  void main(){
    v_uv = (a_pos + 1.0) * 0.5;
    gl_Position = vec4(a_pos, 0.0, 1.0);
  }`;

  const FRAG = `
  precision highp float;
  varying vec2 v_uv;

  uniform sampler2D u_img;
  uniform sampler2D u_mask;
  uniform sampler2D u_curveLUT;

  uniform vec2 u_viewSize;      // canvas px
  uniform vec2 u_imgSize;       // image px
  uniform float u_zoom;
  uniform vec2 u_panPx;         // pan in px (screen space)
  uniform float u_rotateRad;

  uniform float u_exposure;     // stops
  uniform float u_contrast;     // -1..1
  uniform float u_saturation;   // -1..1
  uniform float u_temperature;  // -1..1

  // mask adjustments
  uniform float u_maskExposure;
  uniform float u_maskSaturation;

  // HSL ranges: 8 vec3 (h,s,l)
  uniform vec3 u_hsl0;
  uniform vec3 u_hsl1;
  uniform vec3 u_hsl2;
  uniform vec3 u_hsl3;
  uniform vec3 u_hsl4;
  uniform vec3 u_hsl5;
  uniform vec3 u_hsl6;
  uniform vec3 u_hsl7;

  vec3 rgb2hsv(vec3 c){
    vec4 K = vec4(0.0, -1.0/3.0, 2.0/3.0, -1.0);
    vec4 p = mix(vec4(c.bg, K.wz), vec4(c.gb, K.xy), step(c.b, c.g));
    vec4 q = mix(vec4(p.xyw, c.r), vec4(c.r, p.yzx), step(p.x, c.r));
    float d = q.x - min(q.w, q.y);
    float e = 1.0e-10;
    return vec3(abs(q.z + (q.w - q.y) / (6.0 * d + e)), d / (q.x + e), q.x);
  }

  vec3 hsv2rgb(vec3 c){
    vec3 p = abs(fract(c.xxx + vec3(0.0, 2.0/3.0, 1.0/3.0)) * 6.0 - 3.0);
    vec3 rgb = clamp(p - 1.0, 0.0, 1.0);
    return c.z * mix(vec3(1.0), rgb, c.y);
  }

  float hueWeight(float h, float center, float width){
    // circular distance on [0..1]
    float d = abs(h - center);
    d = min(d, 1.0 - d);
    float w = 1.0 - smoothstep(width*0.5, width, d);
    return w;
  }

  vec3 applyHSLRanges(vec3 rgb){
    vec3 hsv = rgb2hsv(rgb);
    float h = hsv.x;
    float s = hsv.y;
    float v = hsv.z;

    // centers for 8 ranges (approx)
    // red(0), orange(0.083), yellow(0.166), green(0.333), aqua(0.5), blue(0.666), purple(0.75), magenta(0.916)
    float w0 = hueWeight(h, 0.000, 0.18);
    float w1 = hueWeight(h, 0.083, 0.18);
    float w2 = hueWeight(h, 0.166, 0.18);
    float w3 = hueWeight(h, 0.333, 0.20);
    float w4 = hueWeight(h, 0.500, 0.20);
    float w5 = hueWeight(h, 0.666, 0.20);
    float w6 = hueWeight(h, 0.750, 0.18);
    float w7 = hueWeight(h, 0.916, 0.18);

    vec3 a0 = u_hsl0;
    vec3 a1 = u_hsl1;
    vec3 a2 = u_hsl2;
    vec3 a3 = u_hsl3;
    vec3 a4 = u_hsl4;
    vec3 a5 = u_hsl5;
    vec3 a6 = u_hsl6;
    vec3 a7 = u_hsl7;

    float sumW = w0+w1+w2+w3+w4+w5+w6+w7 + 1e-6;
    vec3 adj =
      (a0*w0 + a1*w1 + a2*w2 + a3*w3 + a4*w4 + a5*w5 + a6*w6 + a7*w7) / sumW;

    // adj.x hue shift, adj.y sat, adj.z lum-ish (we push v)
    hsv.x = fract(hsv.x + adj.x * 0.08); // small scale to keep controllable
    hsv.y = clamp(hsv.y * (1.0 + adj.y), 0.0, 1.0);
    hsv.z = clamp(hsv.z * (1.0 + adj.z), 0.0, 1.0);

    return hsv2rgb(hsv);
  }

  vec3 applyBasic(vec3 c, float expStops, float con, float sat){
    // exposure: multiply by 2^stops
    c *= pow(2.0, expStops);

    // contrast around 0.5
    c = (c - 0.5) * (1.0 + con) + 0.5;

    // saturation
    float luma = dot(c, vec3(0.2126, 0.7152, 0.0722));
    c = mix(vec3(luma), c, 1.0 + sat);

    return clamp(c, 0.0, 1.0);
  }

  vec3 applyTemp(vec3 c, float t){
    // simple white balance shift (fast approximation)
    // t -1..1: cool to warm
    vec3 warm = vec3(1.06, 1.0, 0.92);
    vec3 cool = vec3(0.94, 1.0, 1.08);
    vec3 m = mix(cool, warm, (t+1.0)*0.5);
    return clamp(c * m, 0.0, 1.0);
  }

  vec3 applyCurve(vec3 c){
    // LUT is RGBA 256x1, packed as:
    // R=RGB curve, G=R curve, B=G curve, A=B curve
    float ixR = c.r * 255.0;
    float ixG = c.g * 255.0;
    float ixB = c.b * 255.0;

    float xR = floor(ixR);
    float xG = floor(ixG);
    float xB = floor(ixB);

    float fR = fract(ixR);
    float fG = fract(ixG);
    float fB = fract(ixB);

    vec2 uvR0 = vec2((xR + 0.5) / 256.0, 0.5);
    vec2 uvR1 = vec2((min(xR+1.0, 255.0) + 0.5) / 256.0, 0.5);

    vec2 uvG0 = vec2((xG + 0.5) / 256.0, 0.5);
    vec2 uvG1 = vec2((min(xG+1.0, 255.0) + 0.5) / 256.0, 0.5);

    vec2 uvB0 = vec2((xB + 0.5) / 256.0, 0.5);
    vec2 uvB1 = vec2((min(xB+1.0, 255.0) + 0.5) / 256.0, 0.5);

    vec4 tR0 = texture2D(u_curveLUT, uvR0);
    vec4 tR1 = texture2D(u_curveLUT, uvR1);
    vec4 tG0 = texture2D(u_curveLUT, uvG0);
    vec4 tG1 = texture2D(u_curveLUT, uvG1);
    vec4 tB0 = texture2D(u_curveLUT, uvB0);
    vec4 tB1 = texture2D(u_curveLUT, uvB1);

    // RGB curve is .r channel
    float rgbR = mix(tR0.r, tR1.r, fR);
    float rgbG = mix(tG0.r, tG1.r, fG);
    float rgbB = mix(tB0.r, tB1.r, fB);

    // per-channel curves
    float cr = mix(tR0.g, tR1.g, fR);
    float cg = mix(tG0.b, tG1.b, fG);
    float cb = mix(tB0.a, tB1.a, fB);

    vec3 outc = vec3(rgbR, rgbG, rgbB);
    outc = vec3(
      mix(outc.r, cr, 0.65),
      mix(outc.g, cg, 0.65),
      mix(outc.b, cb, 0.65)
    );
    return clamp(outc, 0.0, 1.0);
  }

  void main(){
    // v_uv is screen uv; build a camera transform mapping to image uv
    vec2 p = (v_uv * u_viewSize); // screen px
    vec2 center = u_viewSize * 0.5;

    // Apply pan (screen space), then rotate around center, then scale by zoom.
    vec2 q = p - center - u_panPx;

    float cs = cos(-u_rotateRad);
    float sn = sin(-u_rotateRad);
    q = vec2(q.x*cs - q.y*sn, q.x*sn + q.y*cs);
    q /= u_zoom;

    vec2 imgCenter = u_imgSize * 0.5;
    vec2 imgPx = q + imgCenter;

    vec2 uv = imgPx / u_imgSize;

    // Outside image -> checker dark
    if(uv.x < 0.0 || uv.x > 1.0 || uv.y < 0.0 || uv.y > 1.0){
      float c = step(0.5, fract((p.x+p.y)*0.02)) * 0.02 + 0.02;
      gl_FragColor = vec4(vec3(c), 1.0);
      return;
    }

    vec4 src = texture2D(u_img, uv);
    vec3 c = src.rgb;

    // apply global adjustments
    c = applyTemp(c, u_temperature);
    c = applyBasic(c, u_exposure, u_contrast, u_saturation);

    // apply HSL ranges
    c = applyHSLRanges(c);

    // apply curves LUT
    c = applyCurve(c);

    // mask
    float m = texture2D(u_mask, v_uv).r; // screen-space mask
    if(m > 0.0001){
      vec3 cm = c;
      cm = applyBasic(cm, u_maskExposure, 0.0, u_maskSaturation);
      c = mix(c, cm, clamp(m, 0.0, 1.0));
    }

    gl_FragColor = vec4(c, 1.0);
  }`;

  function initGL(){
    gl = glCanvas.getContext("webgl", { premultipliedAlpha:false, antialias:true, preserveDrawingBuffer:true });
    if(!gl){
      alert("WebGL not available in this browser");
      return;
    }

    program = createProgram(gl, VERT, FRAG);
    gl.useProgram(program);

    // Quad
    quadVBO = gl.createBuffer();
    gl.bindBuffer(gl.ARRAY_BUFFER, quadVBO);
    gl.bufferData(gl.ARRAY_BUFFER, new Float32Array([
      -1,-1,  1,-1, -1, 1,
      -1, 1,  1,-1,  1, 1
    ]), gl.STATIC_DRAW);

    const aPos = gl.getAttribLocation(program, "a_pos");
    gl.enableVertexAttribArray(aPos);
    gl.vertexAttribPointer(aPos, 2, gl.FLOAT, false, 0, 0);

    // Uniforms
    u.u_img = gl.getUniformLocation(program, "u_img");
    u.u_mask = gl.getUniformLocation(program, "u_mask");
    u.u_curveLUT = gl.getUniformLocation(program, "u_curveLUT");

    u.u_viewSize = gl.getUniformLocation(program, "u_viewSize");
    u.u_imgSize = gl.getUniformLocation(program, "u_imgSize");
    u.u_zoom = gl.getUniformLocation(program, "u_zoom");
    u.u_panPx = gl.getUniformLocation(program, "u_panPx");
    u.u_rotateRad = gl.getUniformLocation(program, "u_rotateRad");

    u.u_exposure = gl.getUniformLocation(program, "u_exposure");
    u.u_contrast = gl.getUniformLocation(program, "u_contrast");
    u.u_saturation = gl.getUniformLocation(program, "u_saturation");
    u.u_temperature = gl.getUniformLocation(program, "u_temperature");

    u.u_maskExposure = gl.getUniformLocation(program, "u_maskExposure");
    u.u_maskSaturation = gl.getUniformLocation(program, "u_maskSaturation");

    for(let i=0;i<8;i++){
      u[`u_hsl${i}`] = gl.getUniformLocation(program, `u_hsl${i}`);
    }

    // Textures setup
    texImage = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, texImage);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);

    texMask = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, texMask);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);

    texCurve = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, texCurve);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.LINEAR);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);
    gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);

    // Default mask (blank)
    uploadMaskBlank();

    // Default curves LUT
    rebuildCurveLUT();

    resizeAll();
    render();
  }

  function createShader(gl, type, src){
    const sh = gl.createShader(type);
    gl.shaderSource(sh, src);
    gl.compileShader(sh);
    if(!gl.getShaderParameter(sh, gl.COMPILE_STATUS)){
      console.error(gl.getShaderInfoLog(sh));
      throw new Error("Shader compile failed");
    }
    return sh;
  }

  function createProgram(gl, vsSrc, fsSrc){
    const vs = createShader(gl, gl.VERTEX_SHADER, vsSrc);
    const fs = createShader(gl, gl.FRAGMENT_SHADER, fsSrc);
    const p = gl.createProgram();
    gl.attachShader(p, vs);
    gl.attachShader(p, fs);
    gl.linkProgram(p);
    if(!gl.getProgramParameter(p, gl.LINK_STATUS)){
      console.error(gl.getProgramInfoLog(p));
      throw new Error("Program link failed");
    }
    return p;
  }

  function resizeAll(){
    const rect = viewerStage.getBoundingClientRect();

    const q = qualityMode.value;
    let scale = 1;
    const dpr = window.devicePixelRatio || 1;
    if(q === "high") scale = dpr;
    else if(q === "mid") scale = Math.max(1, Math.min(1.5, dpr));
    else scale = Math.max(1, Math.min(1.5, dpr)); // auto

    const w = Math.max(1, Math.floor(rect.width * scale));
    const h = Math.max(1, Math.floor(rect.height * scale));

    if(glCanvas.width !== w || glCanvas.height !== h){
      glCanvas.width = w; glCanvas.height = h;
      maskCanvas.width = w; maskCanvas.height = h;
      drawMaskOverlay(); // redraw mask view
    }

    if(gl){
      gl.viewport(0,0,glCanvas.width, glCanvas.height);
    }

    // Fix “image loads tiny”: always keep fit-to-view sane when we have an image
    if(state.imgW && state.imgH){
      // do not override user manual zoom if they already zoomed substantially
      // but keep image on screen after resize
      if(state.zoom < 0.02) fitToView();
      clampPan();
    }
    updateHUD();
    syncQuickControls();
  }

  window.addEventListener("resize", () => {
    resizeAll();
    render();
  });

  // ---------- Fit / Pan / Zoom / Rotate ----------
  function fitToView(){
    if(!state.imgW || !state.imgH) return;

    const vw = glCanvas.width;
    const vh = glCanvas.height;

    // Fit image to view accounting for rotation is complex; we fit unrotated
    const sx = vw / state.imgW;
    const sy = vh / state.imgH;
    const fit = Math.min(sx, sy) * 0.95;

    state.zoom = fit;
    state.panX = 0;
    state.panY = 0;
    updateHUD();
    
  syncQuickControls();
  }

  function clampPan(){
    // Keep some boundary so image doesn't disappear fully
    const vw = glCanvas.width;
    const vh = glCanvas.height;
    const imgW = state.imgW * state.zoom;
    const imgH = state.imgH * state.zoom;

    const maxX = Math.max(0, (imgW - vw) * 0.5 + 40);
    const maxY = Math.max(0, (imgH - vh) * 0.5 + 40);

    state.panX = Math.max(-maxX, Math.min(maxX, state.panX));
    state.panY = Math.max(-maxY, Math.min(maxY, state.panY));
  }

  function updateHUD(){
    hudZoom.textContent = `Zoom: ${Math.round(state.zoom * 100)}%`;
    hudRotate.textContent = `Rotate: ${Math.round(state.rotateDeg)}°`;
    const label =
      state.mode === "viewer" ? "Viewer" :
      state.mode === "linear" ? "Linear Mask" :
      state.mode === "radial" ? "Radial Mask" :
      "Erase Mask";
    hudMode.textContent = `Mode: ${label}`;
  }
  function syncQuickControls(){
  // zoom slider uses percent
  const zp = Math.round(state.zoom * 100);
  qcZoom.value = String(Math.max(10, Math.min(400, zp)));

  qcRotate.value = String(Math.round(state.rotateDeg));
}

function setZoomFromPercent(p){
  state.zoom = Math.max(0.02, Math.min(20, p / 100));
  clampPan();
  updateHUD();
  render();
  syncQuickControls();
}

function setRotationDeg(deg){
  state.rotateDeg = Math.max(-180, Math.min(180, deg));
  rotate.value = state.rotateDeg;     // keep side slider in sync
  updateHUD();
  render();
  syncQuickControls();
}

  // ---------- Render ----------
  function render(){
    if(!gl) return;
    gl.clearColor(0,0,0,1);
    gl.clear(gl.COLOR_BUFFER_BIT);

    if(!state.imgW || !state.imgH){
      // no image yet, draw blank
      return;
    }

    gl.useProgram(program);

    // Textures
    gl.activeTexture(gl.TEXTURE0);
    gl.bindTexture(gl.TEXTURE_2D, texImage);
    gl.uniform1i(u.u_img, 0);

    gl.activeTexture(gl.TEXTURE1);
    gl.bindTexture(gl.TEXTURE_2D, texMask);
    gl.uniform1i(u.u_mask, 1);

    gl.activeTexture(gl.TEXTURE2);
    gl.bindTexture(gl.TEXTURE_2D, texCurve);
    gl.uniform1i(u.u_curveLUT, 2);

    // Uniforms
    gl.uniform2f(u.u_viewSize, glCanvas.width, glCanvas.height);
    gl.uniform2f(u.u_imgSize, state.imgW, state.imgH);

    gl.uniform1f(u.u_zoom, state.zoom);
    gl.uniform2f(u.u_panPx, state.panX, state.panY);
    gl.uniform1f(u.u_rotateRad, (state.rotateDeg * Math.PI) / 180);

    gl.uniform1f(u.u_exposure, exposure.value);
    gl.uniform1f(u.u_contrast, contrast.value);
    gl.uniform1f(u.u_saturation, saturation.value);
    gl.uniform1f(u.u_temperature, temperature.value);

    gl.uniform1f(u.u_maskExposure, maskExposure.value);
    gl.uniform1f(u.u_maskSaturation, maskSaturation.value);

    for(let i=0;i<8;i++){
      const a = state.hsl[i];
      gl.uniform3f(u[`u_hsl${i}`], a.h, a.s, a.l);
    }

    gl.drawArrays(gl.TRIANGLES, 0, 6);
  }

  // ---------- Mask handling (screen-space alpha) ----------
  const maskCtx = maskCanvas.getContext("2d");
  const maskData = {
    // store mask points for redraw on resize
    linear: null, // {x0,y0,x1,y1, strength}
    radial: null, // {cx,cy,r, strength}
    erase: []      // not stored; we erase directly
  };

  function uploadMaskBlank(){
    const w = glCanvas.width || 2;
    const h = glCanvas.height || 2;
    const blank = new Uint8Array(w*h);
    gl.bindTexture(gl.TEXTURE_2D, texMask);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.LUMINANCE, w, h, 0, gl.LUMINANCE, gl.UNSIGNED_BYTE, blank);
  }

  function uploadMaskFromCanvas(){
    // read maskCanvas alpha-like from red channel (we draw white on transparent)
    const w = maskCanvas.width, h = maskCanvas.height;
    const img = maskCtx.getImageData(0,0,w,h);
    const out = new Uint8Array(w*h);
    for(let i=0, p=0; i<img.data.length; i+=4, p++){
      out[p] = img.data[i]; // red
    }
    gl.bindTexture(gl.TEXTURE_2D, texMask);
    gl.texImage2D(gl.TEXTURE_2D, 0, gl.LUMINANCE, w, h, 0, gl.LUMINANCE, gl.UNSIGNED_BYTE, out);
  }

  function drawMaskOverlay(){
    maskCtx.clearRect(0,0,maskCanvas.width, maskCanvas.height);

    // subtle visible overlay: draw in cyan-ish white, opacity via alpha
    if(maskData.linear){
      const {x0,y0,x1,y1,strength} = maskData.linear;
      const g = maskCtx.createLinearGradient(x0,y0,x1,y1);
      g.addColorStop(0, `rgba(255,255,255,${0.0})`);
      g.addColorStop(1, `rgba(255,255,255,${0.95*strength})`);
      maskCtx.fillStyle = g;
      maskCtx.fillRect(0,0,maskCanvas.width, maskCanvas.height);
    }
    if(maskData.radial){
      const {cx,cy,r,strength} = maskData.radial;
      const g = maskCtx.createRadialGradient(cx,cy,0,cx,cy,r);
      g.addColorStop(0, `rgba(255,255,255,${0.95*strength})`);
      g.addColorStop(1, `rgba(255,255,255,${0.0})`);
      maskCtx.fillStyle = g;
      maskCtx.fillRect(0,0,maskCanvas.width, maskCanvas.height);
    }

    uploadMaskFromCanvas();
  }

  function clearMask(){
    maskData.linear = null;
    maskData.radial = null;
    maskCtx.clearRect(0,0,maskCanvas.width, maskCanvas.height);
    uploadMaskFromCanvas();
    render();
  }

  btnClearMask.addEventListener("click", () => {
    clearMask();
  });

  // ---------- Curves (Canvas UI) ----------
  const cctx = curveCanvas.getContext("2d");
  const curveUI = {
    draggingIndex: -1,
    hoverIndex: -1,
    padding: 32,
    radius: 10
  };

  function curveToCanvas(pt){
    const pad = curveUI.padding;
    const w = curveCanvas.width;
    const h = curveCanvas.height;
    return {
      x: pad + pt.x * (w - 2*pad),
      y: (h - pad) - pt.y * (h - 2*pad)
    };
  }

  function canvasToCurve(x, y){
    const pad = curveUI.padding;
    const w = curveCanvas.width;
    const h = curveCanvas.height;
    const cx = (x - pad) / (w - 2*pad);
    const cy = ((h - pad) - y) / (h - 2*pad);
    return { x: clamp01(cx), y: clamp01(cy) };
  }

  function getActiveCurvePoints(){
    return state.curves[state.activeCurve];
  }

  function sortPoints(pts){
    pts.sort((a,b) => a.x - b.x);
    // clamp endpoints to x=0 and x=1 if they exist
    pts[0].x = 0;
    pts[pts.length-1].x = 1;
  }

  function drawCurveUI(){
    const w = curveCanvas.width, h = curveCanvas.height;
    cctx.clearRect(0,0,w,h);

    // grid
    cctx.save();
    cctx.globalAlpha = 0.7;
    cctx.strokeStyle = "rgba(255,255,255,0.08)";
    cctx.lineWidth = 1;
    const pad = curveUI.padding;
    for(let i=0;i<=4;i++){
      const t = i/4;
      const x = pad + t*(w-2*pad);
      const y = pad + t*(h-2*pad);
      cctx.beginPath(); cctx.moveTo(x,pad); cctx.lineTo(x,h-pad); cctx.stroke();
      cctx.beginPath(); cctx.moveTo(pad,y); cctx.lineTo(w-pad,y); cctx.stroke();
    }
    // border
    cctx.strokeStyle = "rgba(255,255,255,0.12)";
    cctx.strokeRect(pad,pad,w-2*pad,h-2*pad);
    cctx.restore();

    const pts = getActiveCurvePoints();
    sortPoints(pts);

    // curve polyline (sampled)
    const lut = buildLUT(pts);
    cctx.save();
    cctx.strokeStyle = "rgba(102,166,255,0.9)";
    if(state.activeCurve !== "rgb"){
      // distinct
      cctx.strokeStyle = "rgba(138,107,255,0.9)";
    }
    cctx.lineWidth = 3;
    cctx.beginPath();
    for(let i=0;i<256;i++){
      const x = i/255;
      const y = lut[i];
      const p = curveToCanvas({x,y});
      if(i===0) cctx.moveTo(p.x,p.y);
      else cctx.lineTo(p.x,p.y);
    }
    cctx.stroke();
    cctx.restore();

    // points
    for(let i=0;i<pts.length;i++){
      const p = curveToCanvas(pts[i]);
      const isDrag = (i === curveUI.draggingIndex);
      const isHover = (i === curveUI.hoverIndex);
      cctx.save();
      cctx.fillStyle = isDrag || isHover ? "rgba(255,255,255,0.95)" : "rgba(255,255,255,0.75)";
      cctx.strokeStyle = "rgba(0,0,0,0.4)";
      cctx.lineWidth = 2;
      cctx.beginPath();
      cctx.arc(p.x,p.y, isDrag ? 11 : 9, 0, Math.PI*2);
      cctx.fill();
      cctx.stroke();
      cctx.restore();
    }
  }

  function buildLUT(points){
    // monotone-ish cubic interpolation (simple Catmull-Rom on y with x parameterization)
    const pts = points.slice().sort((a,b)=>a.x-b.x);
    const out = new Float32Array(256);

    function sample(x){
      // find segment
      let i = 0;
      while(i < pts.length-2 && x > pts[i+1].x) i++;
      const p0 = pts[Math.max(0, i-1)];
      const p1 = pts[i];
      const p2 = pts[i+1];
      const p3 = pts[Math.min(pts.length-1, i+2)];

      const t = (x - p1.x) / Math.max(1e-6, (p2.x - p1.x));
      // Catmull-Rom on y
      const t2 = t*t, t3=t2*t;
      const y =
        0.5 * (
          (2*p1.y) +
          (-p0.y + p2.y)*t +
          (2*p0.y - 5*p1.y + 4*p2.y - p3.y)*t2 +
          (-p0.y + 3*p1.y - 3*p2.y + p3.y)*t3
        );
      return clamp01(y);
    }

    for(let i=0;i<256;i++){
      out[i] = sample(i/255);
    }
    return out;
  }

  function rebuildCurveLUT(){
    // pack into RGBA:
    // R = rgb curve
    // G = r curve
    // B = g curve
    // A = b curve
    const rgb = buildLUT(state.curves.rgb);
    const r = buildLUT(state.curves.r);
    const g = buildLUT(state.curves.g);
    const b = buildLUT(state.curves.b);

    const data = new Uint8Array(256 * 4);
    for(let i=0;i<256;i++){
      data[i*4+0] = Math.round(rgb[i]*255);
      data[i*4+1] = Math.round(r[i]*255);
      data[i*4+2] = Math.round(g[i]*255);
      data[i*4+3] = Math.round(b[i]*255);
    }

    if(gl && texCurve){
      gl.bindTexture(gl.TEXTURE_2D, texCurve);
      gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, 256, 1, 0, gl.RGBA, gl.UNSIGNED_BYTE, data);
    }
  }

  // curve interactions
  function hitTestPoint(mx,my){
    const pts = getActiveCurvePoints();
    for(let i=0;i<pts.length;i++){
      const p = curveToCanvas(pts[i]);
      const dx = mx - p.x, dy = my - p.y;
      if(dx*dx + dy*dy <= (curveUI.radius*curveUI.radius)) return i;
    }
    return -1;
  }

  curveCanvas.addEventListener("pointerdown", (e) => {
    curveCanvas.setPointerCapture(e.pointerId);
    const rect = curveCanvas.getBoundingClientRect();
    const mx = (e.clientX - rect.left) * (curveCanvas.width / rect.width);
    const my = (e.clientY - rect.top) * (curveCanvas.height / rect.height);

    const idx = hitTestPoint(mx,my);
    if(idx !== -1){
      curveUI.draggingIndex = idx;
    }else{
      // add point on click (except too close to edges)
      const pt = canvasToCurve(mx,my);
      const pts = getActiveCurvePoints();
      pts.push(pt);
      sortPoints(pts);
      curveUI.draggingIndex = hitTestPoint(mx,my);
      rebuildCurveLUT();
      render();
    }
    drawCurveUI();
  });

  curveCanvas.addEventListener("pointermove", (e) => {
    const rect = curveCanvas.getBoundingClientRect();
    const mx = (e.clientX - rect.left) * (curveCanvas.width / rect.width);
    const my = (e.clientY - rect.top) * (curveCanvas.height / rect.height);

    if(curveUI.draggingIndex !== -1){
      const pt = canvasToCurve(mx,my);
      const pts = getActiveCurvePoints();

      // lock endpoints x, only move y
      if(curveUI.draggingIndex === 0){
        pts[0].y = pt.y;
      } else if(curveUI.draggingIndex === pts.length-1){
        pts[pts.length-1].y = pt.y;
      } else {
        // keep x order by clamping between neighbors
        const prev = pts[curveUI.draggingIndex-1];
        const next = pts[curveUI.draggingIndex+1];
        pts[curveUI.draggingIndex].x = Math.max(prev.x + 0.01, Math.min(next.x - 0.01, pt.x));
        pts[curveUI.draggingIndex].y = pt.y;
      }
      rebuildCurveLUT();
      render();
      drawCurveUI();
    } else {
      curveUI.hoverIndex = hitTestPoint(mx,my);
      drawCurveUI();
    }
  });
   qcZoom.addEventListener("input", () => {
  if(!state.imgW) return;
  setZoomFromPercent(Number(qcZoom.value));
});

qcRotate.addEventListener("input", () => {
  if(!state.imgW) return;
  setRotationDeg(Number(qcRotate.value));
});

qcZoomOut.addEventListener("click", () => {
  if(!state.imgW) return;
  setZoomFromPercent(Math.round(state.zoom * 100 * 0.92));
});

qcZoomIn.addEventListener("click", () => {
  if(!state.imgW) return;
  setZoomFromPercent(Math.round(state.zoom * 100 * 1.08));
});

qcFit.addEventListener("click", () => {
  if(!state.imgW) return;
  fitToView();
  updateHUD();
  render();
  syncQuickControls();
});

qcRotL.addEventListener("click", () => {
  if(!state.imgW) return;
  setRotationDeg(state.rotateDeg - 90);
});

qcRotR.addEventListener("click", () => {
  if(!state.imgW) return;
  setRotationDeg(state.rotateDeg + 90);
});

qcRot0.addEventListener("click", () => {
  if(!state.imgW) return;
  setRotationDeg(0);
});

  curveCanvas.addEventListener("pointerup", (e) => {
    curveUI.draggingIndex = -1;
    drawCurveUI();
  });

  btnCurveAdd.addEventListener("click", () => {
    const pts = getActiveCurvePoints();
    pts.push({x:0.5, y:0.5});
    sortPoints(pts);
    rebuildCurveLUT();
    render();
    drawCurveUI();
  });

  btnCurveReset.addEventListener("click", () => {
    state.curves[state.activeCurve] = defaultCurvePoints();
    rebuildCurveLUT();
    render();
    drawCurveUI();
  });

  curveTabs.forEach(btn => {
    btn.addEventListener("click", () => {
      curveTabs.forEach(b => b.classList.remove("active"));
      btn.classList.add("active");
      state.activeCurve = btn.dataset.curve;
      drawCurveUI();
    });
  });

  // ---------- HSL UI ----------
  function setHslActive(index){
    state.hslActive = index;
    const v = state.hsl[index];
    hslHue.value = v.h;
    hslSat.value = v.s;
    hslLum.value = v.l;

    hslTabs.forEach((t,i)=> t.classList.toggle("active", i===index));
  }

  hslTabs.forEach((t, i) => {
    t.addEventListener("click", () => setHslActive(i));
  });

  function updateHslFromSliders(){
    const v = state.hsl[state.hslActive];
    v.h = hslHue.value;
    v.s = hslSat.value;
    v.l = hslLum.value;
    render();
  }

  hslHue.onChange(updateHslFromSliders);
  hslSat.onChange(updateHslFromSliders);
  hslLum.onChange(updateHslFromSliders);

  // ---------- Mode switching ----------
  function setMode(mode){
    state.mode = mode;
    modeViewerBtn.classList.toggle("active", mode==="viewer");
    modeLinearBtn.classList.toggle("active", mode==="linear");
    modeRadialBtn.classList.toggle("active", mode==="radial");
    modeEraseBtn.classList.toggle("active", mode==="erase");
    updateHUD();

    // mask canvas pointer events only in mask modes
    maskCanvas.style.pointerEvents = (mode === "viewer") ? "none" : "auto";
  }

  modeViewerBtn.addEventListener("click", () => setMode("viewer"));
  modeLinearBtn.addEventListener("click", () => setMode("linear"));
  modeRadialBtn.addEventListener("click", () => setMode("radial"));
  modeEraseBtn.addEventListener("click", () => setMode("erase"));

  // ---------- Viewer interactions (pan/zoom/rotate + masks) ----------
  let dragging = false;
  let startX=0, startY=0;
  let startPanX=0, startPanY=0;

  glCanvas.addEventListener("pointerdown", (e) => {
    glCanvas.setPointerCapture(e.pointerId);
    dragging = true;
    startX = e.clientX; startY = e.clientY;
    startPanX = state.panX; startPanY = state.panY;
  });

  glCanvas.addEventListener("pointermove", (e) => {
    if(!dragging) return;
    if(state.mode !== "viewer") return;

    state.panX = startPanX + (e.clientX - startX) * (glCanvas.width / viewerStage.clientWidth);
    state.panY = startPanY + (e.clientY - startY) * (glCanvas.height / viewerStage.clientHeight);
    clampPan();
    render();
  });

  glCanvas.addEventListener("pointerup", () => {
    dragging = false;
  });

  viewerStage.addEventListener("wheel", (e) => {
    if(!state.imgW) return;
    e.preventDefault();
    const delta = Math.sign(e.deltaY);

    if(wheelMode.value === "rotate"){
      state.rotateDeg = Math.max(-180, Math.min(180, state.rotateDeg + delta * 2));
      rotate.value = state.rotateDeg;
    } else {
      // zoom around
      const factor = (delta > 0) ? 0.92 : 1.08;
      state.zoom = Math.max(0.02, Math.min(20, state.zoom * factor));
      clampPan();
    }
    updateHUD();
    render();
  }, { passive:false });

  // Mask drawing (linear/radial/erase)
  let maskDrag = false;
  let maskStart = null;

  maskCanvas.addEventListener("pointerdown", (e) => {
    if(state.mode === "viewer") return;
    maskCanvas.setPointerCapture(e.pointerId);
    maskDrag = true;
    maskStart = { x: e.offsetX * (maskCanvas.width / maskCanvas.clientWidth), y: e.offsetY * (maskCanvas.height / maskCanvas.clientHeight) };
  });

  maskCanvas.addEventListener("pointermove", (e) => {
    if(!maskDrag) return;
    const x = e.offsetX * (maskCanvas.width / maskCanvas.clientWidth);
    const y = e.offsetY * (maskCanvas.height / maskCanvas.clientHeight);

    if(state.mode === "linear"){
      maskData.linear = { x0: maskStart.x, y0: maskStart.y, x1: x, y1: y, strength: 1.0 };
      drawMaskOverlay();
      render();
    } else if(state.mode === "radial"){
      const dx = x - maskStart.x, dy = y - maskStart.y;
      const r = Math.max(10, Math.sqrt(dx*dx + dy*dy));
      maskData.radial = { cx: maskStart.x, cy: maskStart.y, r, strength: 1.0 };
      drawMaskOverlay();
      render();
    } else if(state.mode === "erase"){
      // erase in brush-like manner
      maskCtx.save();
      maskCtx.globalCompositeOperation = "destination-out";
      maskCtx.fillStyle = "rgba(0,0,0,0.85)";
      maskCtx.beginPath();
      maskCtx.arc(x,y, 40, 0, Math.PI*2);
      maskCtx.fill();
      maskCtx.restore();
      uploadMaskFromCanvas();
      render();
    }
  });

  maskCanvas.addEventListener("pointerup", () => {
    maskDrag = false;
    maskStart = null;
  });

  // ---------- Side panel behavior ----------
  function setPanel(open){
    rightPanel.classList.toggle("closed", !open);
    btnTogglePanel.setAttribute("aria-expanded", String(open));
  }

  btnTogglePanel.addEventListener("click", () => {
    const open = rightPanel.classList.contains("closed");
    setPanel(open);
  });
  btnClosePanel.addEventListener("click", () => setPanel(false));

  window.addEventListener("keydown", (e) => {
    if(e.key === "Escape"){
      setPanel(false);
    }
  });

  // ---------- Slider reactions ----------
  exposure.onChange(() => render());
  contrast.onChange(() => render());
  saturation.onChange(() => render());
  temperature.onChange(() => render());
  rotate.onChange((v) => {
    state.rotateDeg = v;
    updateHUD();
    render();
  });

  maskExposure.onChange(() => render());
  maskSaturation.onChange(() => render());

  qualityMode.addEventListener("change", () => {
    resizeAll();
    render();
  });

  btnFit.addEventListener("click", () => {
    fitToView();
    render();
  });

  btnReset.addEventListener("click", () => {
    // Reset globals
    exposure.value = 0;
    contrast.value = 0;
    saturation.value = 0;
    temperature.value = 0;
    rotate.value = 0;
    state.rotateDeg = 0;

    // reset HSL
    state.hsl = Array.from({length:8}, () => ({ h:0, s:0, l:0 }));
    setHslActive(state.hslActive);

    // reset curves
    state.curves.rgb = defaultCurvePoints();
    state.curves.r = defaultCurvePoints();
    state.curves.g = defaultCurvePoints();
    state.curves.b = defaultCurvePoints();
    rebuildCurveLUT();
    drawCurveUI();

    // reset mask adjustments + mask itself
    maskExposure.value = 0;
    maskSaturation.value = 0;
    clearMask();

    // reset view
    fitToView();
    render();
  });

  // ---------- Open / Decode ----------
  btnOpen.addEventListener("click", () => fileInput.click());
  btnOpen2.addEventListener("click", () => fileInput.click());

fileInput.addEventListener("change", async () => {
  const f = fileInput.files && fileInput.files[0];
  if(!f) return;

  state.fileName = f.name;
  hudName.textContent = f.name;

  const raw = isRawLike(f.name);
  detectedType.textContent = raw ? "RAW/DNG" : (f.type || "Image");

  if(raw){
    rawStatus.querySelector(".infoText").innerHTML =
      "RAW selected. To enable decoding, add a WASM RAW decoder in <b>libs/libraw</b><br/>" +
      "If not installed, export will not work for RAW";
    const ok = await tryDecodeRawWithLibRaw(f);
    if(!ok){
      alert("RAW decoder not found. Add libs/libraw/libraw.js and libs/libraw/libraw.wasm");
    }
    return;
  }

  try{
    // Apply EXIF orientation automatically
let bmp;
try {
  bmp = await createImageBitmap(f, { imageOrientation: "from-image" });
} catch {
  bmp = await createImageBitmap(f);
}
await setImageBitmap(bmp);
  } catch(err){
    console.error(err);
    alert("Failed to load image");
  }
});


 async function setImageBitmap(bmp){
  imgBitmap = bmp;
  state.imgW = bmp.width;
  state.imgH = bmp.height;
  hudInfo.textContent = `${state.imgW}×${state.imgH}`;

  // upload to GL texture (FIX: flip Y)
  gl.bindTexture(gl.TEXTURE_2D, texImage);

  gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true);
  gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, bmp);
  gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, false);

  fitToView();
  syncQuickControls();
  clearMask();
  drawCurveUI();
  updateHUD();
  render();
}

  // ---------- Export ----------
  btnExport.addEventListener("click", () => {
    if(!state.imgW) return;

    // We can export current view render from glCanvas directly as PNG
    // This exports the viewport size output (not full resolution)
    glCanvas.toBlob((blob) => {
      if(!blob) return;
      const a = document.createElement("a");
      a.href = URL.createObjectURL(blob);
      a.download = (state.fileName ? state.fileName.replace(/\.[^/.]+$/, "") : "export") + "_edit.png";
      a.click();
      setTimeout(() => URL.revokeObjectURL(a.href), 1000);
    }, "image/png");
  });

  // ---------- RAW support placeholder ----------
  // This is a plug-in point. Many LibRaw WASM builds expose a function that takes bytes -> RGB buffer.
  // You will need to integrate the specific build you choose.
  async function tryDecodeRawWithLibRaw(file){
    // Detect presence of libs/libraw/libraw.js
    // If present, it should attach something like window.LibRaw or window.librawModule
    let has = false;
    try{
      has = await probeScript("./libs/libraw/libraw.js");
    } catch(e){
      has = false;
    }
    if(!has) return false;

    // Load script
    await loadScriptOnce("./libs/libraw/libraw.js");

    // Example placeholder:
    // If your libraw.js exposes `createLibRaw()` returning { decodeToImageData(bytes) }
    const bytes = new Uint8Array(await file.arrayBuffer());

    if(typeof window.createLibRaw !== "function"){
      console.warn("libraw.js loaded but createLibRaw() not found. Wire it to your build.");
      return false;
    }

    try{
      const libraw = await window.createLibRaw({ wasmPath: "./libs/libraw/libraw.wasm" });
      const decoded = await libraw.decodeToImageData(bytes); // expects {width,height,data:Uint8ClampedArray RGBA}
      const bmp = await imageDataToBitmap(decoded);
      await setImageBitmap(bmp);
      return true;
    } catch(err){
      console.error(err);
      return false;
    }
  }

  function imageDataToBitmap(decoded){
    return new Promise((resolve) => {
      const cnv = document.createElement("canvas");
      cnv.width = decoded.width;
      cnv.height = decoded.height;
      const ctx = cnv.getContext("2d");
      const img = new ImageData(decoded.data, decoded.width, decoded.height);
      ctx.putImageData(img, 0, 0);
      cnv.toBlob(async (blob) => {
        const bmp = await createImageBitmap(blob);
        resolve(bmp);
      });
    });
  }

  function loadScriptOnce(src){
    return new Promise((resolve, reject) => {
      const existing = document.querySelector(`script[data-src="${src}"]`);
      if(existing) return resolve();

      const s = document.createElement("script");
      s.src = src;
      s.async = true;
      s.dataset.src = src;
      s.onload = () => resolve();
      s.onerror = () => reject(new Error("Failed to load script: " + src));
      document.head.appendChild(s);
    });
  }

  async function probeScript(src){
    const r = await fetch(src, { method:"GET" });
    return r.ok;
  }

  // ---------- Start ----------
  initGL();
  setPanel(true);
  setMode("viewer");
  setHslActive(0);
  drawCurveUI();

})();
