(() => {
  // =========================================================
  // OLD BROWSER GUARD (include this in all future web apps)
  // =========================================================
  function browserIsTooOld(){
    try{
      return !(
        'Promise' in window &&
        'Map' in window &&
        'Set' in window &&
        'fetch' in window &&
        'localStorage' in window &&
        'FileReader' in window &&
        'Blob' in window &&
        'URL' in window &&
        window.CSS && CSS.supports && CSS.supports('color', 'var(--x)')
      )
    }catch(e){
      return true
    }
  }

  const $ = (id) => document.getElementById(id)

  // UI elements
  const docEl = $('doc')
  const pageListEl = $('pageList')

  const fileOpen = $('fileOpen')
  const fileImage = $('fileImage')

  const statusText = $('statusText')
  const badgeRO = $('badgeRO')

  const modal = $('modal')
  const modalTitle = $('modalTitle')
  const modalBody = $('modalBody')
  const modalClose = $('modalClose')

  // State
  const LS_KEY = 'freehtmlapp_word_pages_v1'
  let currentName = 'freehtml.app_document'
  let readOnly = false
  let activePageIndex = 0

  function showModal(title, html){
    modalTitle.textContent = title
    modalBody.innerHTML = html
    modal.style.display = 'flex'
  }
  modalClose.onclick = () => modal.style.display = 'none'
  modal.onclick = (e) => { if(e.target === modal) modal.style.display = 'none' }

  function setStatus(msg){
    statusText.textContent = msg
    clearTimeout(setStatus._t)
    setStatus._t = setTimeout(() => statusText.textContent = 'Ready', 900)
  }

  function setReadOnly(flag){
    readOnly = !!flag
    badgeRO.style.display = readOnly ? 'inline-block' : 'none'
    getAllBodies().forEach(b => b.contentEditable = readOnly ? 'false' : 'true')
    $('btnAddPage').disabled = readOnly
    $('btnDeletePage').disabled = readOnly
  }

  if(browserIsTooOld()){
    showModal(
      'Browser not supported',
      `Your browser is not compatible with this web app<br><br>
       Please download the latest browser: Google Chrome, Microsoft Edge, or Mozilla Firefox<br><br>
       After updating, reopen this page`
    )
    document.querySelectorAll('button, input, select, textarea').forEach(x => x.disabled = true)
    return
  }

  // Dependency warnings
  if(typeof docx === 'undefined'){
    showModal('DOCX export library missing', `Missing <b>./js/docx.umd.js</b><br><br>DOCX export will not work`)
  }
  if(typeof mammoth === 'undefined'){
    showModal('DOCX open/import library missing', `Missing <b>./js/mammoth.browser.min.js</b><br><br>DOCX open will not work`)
  }
  if(typeof html2pdf === 'undefined'){
    showModal('PDF export library missing', `Missing <b>./js/html2pdf.bundle.min.js</b><br><br>PDF export will not work`)
  }

  // =========================================================
  // Pages model in DOM (each page has .pageBody)
  // =========================================================
  function getPages(){ return Array.from(docEl.querySelectorAll('.page')) }
  function getAllBodies(){ return Array.from(docEl.querySelectorAll('.pageBody')) }
  function getBody(i){ return getAllBodies()[i] || null }

  function createPage(pageNum){
    const page = document.createElement('div')
    page.className = 'page'
    page.innerHTML = `
      <div class="pageHeader">Page ${pageNum}</div>
      <div class="pageBody" contenteditable="true" spellcheck="false"></div>
    `.trim()
    return page
  }

  function renumberPages(){
    getPages().forEach((p, i) => {
      const h = p.querySelector('.pageHeader')
      if(h) h.textContent = `Page ${i + 1}`
    })
  }

  function ensureAtLeastOnePage(){
    if(getPages().length) return
    docEl.appendChild(createPage(1))
    renumberPages()
    wirePageEvents()
  }

  function clearAllPages(){
    docEl.innerHTML = ''
    ensureAtLeastOnePage()
  }

  function addPage(afterIndex){
    const pages = getPages()
    const newPage = createPage(pages.length + 1)
    if(afterIndex >= pages.length - 1){
      docEl.appendChild(newPage)
    }else{
      pages[afterIndex].insertAdjacentElement('afterend', newPage)
    }
    renumberPages()
    wirePageEvents()
    renderSidebar()
  }

  function deletePage(index){
    const pages = getPages()
    if(pages.length === 1) return
    pages[index].remove()
    activePageIndex = Math.max(0, Math.min(activePageIndex, getPages().length - 1))
    renumberPages()
    renderSidebar()
    focusPage(activePageIndex)
  }

  function focusPage(i){
    const b = getBody(i)
    if(!b) return
    b.focus()
    b.scrollIntoView({behavior:'smooth', block:'center'})
  }

  // =========================================================
  // Sidebar document view
  // =========================================================
  function textSnippet(html){
    const tmp = document.createElement('div')
    tmp.innerHTML = html || ''
    const t = (tmp.textContent || '').trim().replace(/\s+/g,' ')
    return t ? t.slice(0, 70) : '(empty)'
  }

  function renderSidebar(){
    const pages = getPages()
    pageListEl.innerHTML = ''
    pages.forEach((p, i) => {
      const body = p.querySelector('.pageBody')
      const item = document.createElement('div')
      item.className = 'pageItem' + (i === activePageIndex ? ' active' : '')
      item.innerHTML = `
        <div class="pnum">${i+1}</div>
        <div class="psnip">${escapeHtml(textSnippet(body?.innerHTML || ''))}</div>
      `
      item.addEventListener('click', () => {
        activePageIndex = i
        renderSidebar()
        focusPage(i)
      })
      pageListEl.appendChild(item)
    })
  }

  // =========================================================
  // Pagination engine (best-effort)
  // Moves whole blocks (p, h1, ul, table, etc) to keep pages within height
  // =========================================================
  function isBlockNode(n){
    if(!n || n.nodeType !== Node.ELEMENT_NODE) return false
    const tag = n.tagName.toLowerCase()
    return ['p','div','h1','h2','h3','h4','ul','ol','table','blockquote','pre'].includes(tag)
  }

  function normalizePage(body){
    // Wrap stray text nodes into <p>
    const nodes = Array.from(body.childNodes)
    for(const n of nodes){
      if(n.nodeType === Node.TEXT_NODE){
        const t = (n.textContent || '').trim()
        if(!t){
          n.remove()
        }else{
          const p = document.createElement('p')
          p.textContent = t
          body.replaceChild(p, n)
        }
      }
    }
    if(body.childNodes.length === 0){
      const p = document.createElement('p')
      p.innerHTML = '<br>'
      body.appendChild(p)
    }
  }

  function ensureNextPage(i){
    const pages = getPages()
    if(i + 1 < pages.length) return
    addPage(i)
  }

  function moveLastBlockToNext(i){
    const body = getBody(i)
    if(!body) return false
    normalizePage(body)
    ensureNextPage(i)

    const next = getBody(i + 1)
    if(!next) return false
    normalizePage(next)

    // Prefer last element child
    let node = body.lastElementChild
    if(!node){
      // if only text nodes remain
      node = body.lastChild
    }
    if(!node) return false

    // If it's a huge single paragraph, split it
    if(node.nodeType === Node.ELEMENT_NODE && ['p','div'].includes(node.tagName.toLowerCase())){
      const didSplit = splitBlockIfNeeded(body, node, next)
      if(didSplit) return true
    }

    next.insertBefore(node, next.firstChild)
    if(body.childNodes.length === 0){
      const p = document.createElement('p')
      p.innerHTML = '<br>'
      body.appendChild(p)
    }
    return true
  }

  function splitBlockIfNeeded(body, block, nextBody){
    const text = (block.textContent || '').trim()
    if(text.length < 400) return false

    // split by words (binary search)
    const words = text.split(/\s+/)
    if(words.length < 60) return false

    // create two blocks
    const a = document.createElement(block.tagName.toLowerCase())
    const b = document.createElement(block.tagName.toLowerCase())

    let lo = 10
    let hi = words.length - 10
    let best = null

    // try to find a split where first fits
    while(lo <= hi){
      const mid = Math.floor((lo + hi) / 2)
      a.textContent = words.slice(0, mid).join(' ')
      b.textContent = words.slice(mid).join(' ')

      // temporary replace
      const prev = block
      body.replaceChild(a, prev)
      if(body.scrollHeight <= body.clientHeight){
        best = mid
        lo = mid + 1
      }else{
        hi = mid - 1
      }
    }

    // restore original if failed
    if(best === null){
      body.replaceChild(block, a)
      return false
    }

    // finalize split at best
    a.textContent = words.slice(0, best).join(' ')
    b.textContent = words.slice(best).join(' ')
    nextBody.insertBefore(b, nextBody.firstChild)
    return true
  }

  function tryPullFromNext(i){
    const body = getBody(i)
    const next = getBody(i+1)
    if(!body || !next) return false

    normalizePage(body)
    normalizePage(next)

    const first = next.firstElementChild || next.firstChild
    if(!first) return false

    // Try move and check fit
    body.appendChild(first)
    if(body.scrollHeight <= body.clientHeight){
      // success
      if(next.childNodes.length === 0){
        const p = document.createElement('p')
        p.innerHTML = '<br>'
        next.appendChild(p)
      }
      return true
    }

    // revert
    next.insertBefore(first, next.firstChild)
    return false
  }

  function cleanupTrailingEmptyPages(){
    const pages = getPages()
    for(let i=pages.length - 1; i>=1; i--){
      const b = getBody(i)
      const t = (b?.textContent || '').replace(/\s+/g,'').trim()
      const hasMeaning = t.length > 0
      if(!hasMeaning){
        pages[i].remove()
      }else{
        break
      }
    }
    renumberPages()
  }

  function reflowFrom(startIndex){
    const pages = getPages()
    const maxIterations = 300

    // forward overflow push
    let iter = 0
    for(let i=startIndex; i<getPages().length && iter < maxIterations; i++){
      const b = getBody(i)
      if(!b) continue
      normalizePage(b)

      while(b.scrollHeight > b.clientHeight + 1 && iter < maxIterations){
        const moved = moveLastBlockToNext(i)
        if(!moved) break
        iter++
      }
    }

    // pull back to fill gaps
    iter = 0
    for(let i=0; i<getPages().length - 1 && iter < maxIterations; i++){
      let movedAny = false
      while(tryPullFromNext(i) && iter < maxIterations){
        movedAny = true
        iter++
      }
      if(movedAny){
        // after pulling, the next page may now be empty
        cleanupTrailingEmptyPages()
      }
    }

    cleanupTrailingEmptyPages()
    renderSidebar()
    saveLocal()
  }

  // =========================================================
  // Editing helpers
  // =========================================================
  function getActiveBody(){
    return getBody(activePageIndex) || getAllBodies()[0] || null
  }

  function cmd(name, value=null){
    if(readOnly){ setStatus('Read only'); return }
    const body = getActiveBody()
    if(!body) return
    body.focus()
    document.execCommand(name, false, value)
    saveLocal()
    reflowFrom(activePageIndex)
  }

  function insertHTML(html){
    if(readOnly){ setStatus('Read only'); return }
    const body = getActiveBody()
    if(!body) return
    body.focus()
    document.execCommand('insertHTML', false, html)
    saveLocal()
    reflowFrom(activePageIndex)
  }

  // Toolbar buttons
  document.querySelectorAll('button[data-cmd]').forEach(btn => {
    btn.addEventListener('click', () => cmd(btn.dataset.cmd))
  })

  $('btnBullets').onclick = () => cmd('insertUnorderedList')
  $('btnNumbers').onclick = () => cmd('insertOrderedList')

  $('blockStyle').onchange = (e) => {
    const v = e.target.value
    if(v === 'p') cmd('formatBlock', 'P')
    if(v === 'h1') cmd('formatBlock', 'H1')
    if(v === 'h2') cmd('formatBlock', 'H2')
    if(v === 'h3') cmd('formatBlock', 'H3')
  }

  $('btnTable').onclick = () => {
    if(readOnly){ setStatus('Read only'); return }
    const rows = Math.max(1, Math.min(20, parseInt(prompt('Rows', '3') || '3', 10)))
    const cols = Math.max(1, Math.min(10, parseInt(prompt('Columns', '3') || '3', 10)))
    let html = '<table><tbody>'
    for(let r=0;r<rows;r++){
      html += '<tr>'
      for(let c=0;c<cols;c++) html += '<td>&nbsp;</td>'
      html += '</tr>'
    }
    html += '</tbody></table><p></p>'
    insertHTML(html)
    setStatus('Table inserted')
  }

  $('btnImage').onclick = () => fileImage.click()
  fileImage.onchange = async () => {
    if(readOnly){ setStatus('Read only'); return }
    const f = fileImage.files?.[0]
    if(!f) return
    const dataUrl = await fileToDataURL(f)
    insertHTML(`<p><img src="${dataUrl}" style="max-width:100%;height:auto;border:1px solid #d6dde8;border-radius:12px" /></p><p></p>`)
    setStatus('Image inserted')
    fileImage.value = ''
  }

  function fileToDataURL(file){
    return new Promise((resolve,reject) => {
      const fr = new FileReader()
      fr.onload = () => resolve(fr.result)
      fr.onerror = reject
      fr.readAsDataURL(file)
    })
  }

  // =========================================================
  // Page sidebar actions
  // =========================================================
  $('btnAddPage').onclick = () => {
    if(readOnly){ setStatus('Read only'); return }
    addPage(getPages().length - 1)
    activePageIndex = getPages().length - 1
    renderSidebar()
    focusPage(activePageIndex)
    saveLocal()
    setStatus('Page added')
  }

  $('btnDeletePage').onclick = () => {
    if(readOnly){ setStatus('Read only'); return }
    const ok = confirm(`Delete Page ${activePageIndex + 1}?`)
    if(!ok) return
    deletePage(activePageIndex)
    saveLocal()
    setStatus('Page deleted')
  }

  // =========================================================
  // File: New / Open
  // =========================================================
  $('btnNew').onclick = () => {
    const ok = confirm('Create a new document? Unsaved changes will be overwritten')
    if(!ok) return
    currentName = 'freehtml.app_document'
    setReadOnly(false)
    clearAllPages()
    setDefaultDoc()
    activePageIndex = 0
    renderSidebar()
    saveLocal()
    setStatus('New document')
  }

  $('btnOpen').onclick = () => fileOpen.click()

  fileOpen.onchange = async () => {
    const f = fileOpen.files?.[0]
    if(!f) return
    const ext = (f.name.split('.').pop() || '').toLowerCase()
    currentName = f.name.replace(/\.[^.]+$/, '') || 'freehtml.app_document'

    try{
      if(ext === 'txt'){
        const text = await f.text()
        setReadOnly(false)
        clearAllPages()
        getBody(0).textContent = text
        reflowFrom(0)
        setStatus('Opened TXT')
      }else if(ext === 'html' || ext === 'htm'){
        const html = await f.text()
        setReadOnly(false)
        clearAllPages()
        getBody(0).innerHTML = sanitizeImportedHtml(html)
        reflowFrom(0)
        setStatus('Opened HTML')
      }else if(ext === 'docx'){
        if(typeof mammoth === 'undefined'){
          showModal('DOCX open not available', `Missing <b>./js/mammoth.browser.min.js</b>`)
          return
        }
        const buf = await f.arrayBuffer()
        const result = await mammoth.convertToHtml({arrayBuffer: buf})
        const messages = (result.messages || [])

        setReadOnly(false)
        clearAllPages()
        getBody(0).innerHTML = sanitizeImportedHtml(result.value || '')
        activePageIndex = 0
        reflowFrom(0)
        renderSidebar()

        if(messages.length){
          showModal(
            'Opened with limitations',
            `DOCX was converted to HTML for editing. Some Word features may not convert perfectly<br><br>
             Notes<br>
             <ul style="margin:8px 0 0 18px">${messages.slice(0,12).map(m => `<li>${escapeHtml(m.message || String(m))}</li>`).join('')}</ul>`
          )
        }
        setStatus('Opened DOCX')
      }else{
        showModal('Unsupported file', 'Please open DOCX, HTML, or TXT')
      }

      saveLocal()
    }catch(err){
      showModal('Open failed', 'Unable to open this file in the browser. Please try another file')
    }finally{
      fileOpen.value = ''
    }
  }

  function sanitizeImportedHtml(html){
    return String(html)
      .replace(/<script[\s\S]*?>[\s\S]*?<\/script>/gi, '')
      .replace(/on\w+="[^"]*"/gi, '')
      .replace(/on\w+='[^']*'/gi, '')
  }

  function escapeHtml(s){
    return String(s)
      .replaceAll('&','&amp;')
      .replaceAll('<','&lt;')
      .replaceAll('>','&gt;')
      .replaceAll('"','&quot;')
      .replaceAll("'","&#039;")
  }

  function setDefaultDoc(){
    const b = getBody(0)
    b.innerHTML = `
      <h1>freehtml.app Word Clone</h1>
      <p>This is a free, page-based Word-style editor that runs in your browser with no install</p>
      <p>Use the left sidebar to jump between pages and use Convert PDF or Print when needed</p>
      <p><br></p>
    `.trim()
    reflowFrom(0)
  }

  // =========================================================
  // Save DOC (HTML in .doc container)
  // =========================================================
  $('btnSaveDoc').onclick = () => {
    try{
      const html = buildDocHtml(getFullHtml())
      const blob = new Blob([html], {type: 'application/msword;charset=utf-8'})
      downloadBlob(blob, `${safeName()}.doc`)
      setStatus('Saved DOC')
    }catch(e){
      showModal('Save DOC failed', 'Unable to export DOC')
    }
  }

  function buildDocHtml(bodyHtml){
    const css = `
      body{font-family:Calibri,Arial,sans-serif;font-size:11pt;}
      h1{font-size:22pt;} h2{font-size:16pt;} h3{font-size:13pt;}
      table{border-collapse:collapse;width:100%;}
      td,th{border:1px solid #d6dde8;padding:6px;vertical-align:top;}
      img{max-width:100%;height:auto;}
      .pageBreak{page-break-after:always;}
    `
    return `
      <html xmlns:o="urn:schemas-microsoft-com:office:office"
            xmlns:w="urn:schemas-microsoft-com:office:word"
            xmlns="http://www.w3.org/TR/REC-html40">
      <head><meta charset="utf-8"><style>${css}</style></head>
      <body>${bodyHtml}</body></html>
    `.trim()
  }

  // =========================================================
  // Save DOCX (best-effort)
  // =========================================================
  $('btnSaveDocx').onclick = async () => {
    try{
      if(typeof docx === 'undefined'){
        showModal('DOCX export not available', 'Missing ./js/docx.umd.js')
        return
      }

      const doc = buildDocxFromPages()
      const blob = await docx.Packer.toBlob(doc)

      if(typeof saveAs !== 'undefined') saveAs(blob, `${safeName()}.docx`)
      else downloadBlob(blob, `${safeName()}.docx`)

      setStatus('Saved DOCX')
    }catch(err){
      showModal(
        'Save DOCX failed',
        `This web app exports DOCX in best-effort mode. Complex Word features may not export perfectly<br><br>
         For full fidelity, use Microsoft Word`
      )
    }
  }

  function buildDocxFromPages(){
    const {
      Document, Paragraph, TextRun, HeadingLevel,
      AlignmentType, Table, TableRow, TableCell, WidthType, ImageRun, PageBreak
    } = docx

    const numberingConfig = {
      config: [
        {
          reference: "freehtml_bullets",
          levels: [{ level: 0, format: "bullet", text: "•", alignment: AlignmentType.LEFT }],
        },
        {
          reference: "freehtml_numbers",
          levels: [{ level: 0, format: "decimal", text: "%1.", alignment: AlignmentType.LEFT }],
        },
      ],
    }

    const blocks = []
    const bodies = getAllBodies()

    bodies.forEach((body, pageIdx) => {
      blocks.push(...convertContainer(body, {Paragraph, TextRun, HeadingLevel, AlignmentType, Table, TableRow, TableCell, WidthType, ImageRun}))
      if(pageIdx < bodies.length - 1){
        // page break between pages
        blocks.push(new Paragraph({ children: [new PageBreak()] }))
      }
    })

    if(!blocks.length) blocks.push(new Paragraph(""))

    return new Document({
      numbering: numberingConfig,
      sections: [{ properties: {}, children: blocks }],
    })
  }

  function convertContainer(container, api){
    const out = []
    for(const node of Array.from(container.childNodes)){
      out.push(...convertBlock(node, api))
    }
    return out
  }

  function convertBlock(node, api){
    const { Paragraph, TextRun, HeadingLevel, AlignmentType, Table, TableRow, TableCell, WidthType, ImageRun } = api

    if(node.nodeType === Node.TEXT_NODE){
      const t = (node.textContent || '').trim()
      if(!t) return []
      return [new Paragraph({ children: [new TextRun(t)] })]
    }
    if(node.nodeType !== Node.ELEMENT_NODE) return []

    const el = node
    const tag = el.tagName.toLowerCase()

    if(tag === 'h1' || tag === 'h2' || tag === 'h3'){
      const lvl = tag === 'h1' ? HeadingLevel.HEADING_1 : (tag === 'h2' ? HeadingLevel.HEADING_2 : HeadingLevel.HEADING_3)
      return [new Paragraph({ heading: lvl, children: inlineRuns(el, api), alignment: AlignmentType.LEFT })]
    }

    if(tag === 'p' || tag === 'div'){
      return [new Paragraph({ children: inlineRuns(el, api), alignment: AlignmentType.LEFT })]
    }

    if(tag === 'ul' || tag === 'ol'){
      const out = []
      const ref = tag === 'ul' ? "freehtml_bullets" : "freehtml_numbers"
      for(const li of Array.from(el.children)){
        if(li.tagName && li.tagName.toLowerCase() === 'li'){
          out.push(new Paragraph({ children: inlineRuns(li, api), numbering: { reference: ref, level: 0 } }))
        }
      }
      out.push(new Paragraph(""))
      return out
    }

    if(tag === 'table'){
      const rows = []
      const trs = el.querySelectorAll('tr')
      trs.forEach(tr => {
        const cells = []
        tr.querySelectorAll('th,td').forEach(td => {
          cells.push(new TableCell({
            width: { size: 100 / Math.max(1, tr.children.length), type: WidthType.PERCENTAGE },
            children: [new Paragraph({ children: inlineRuns(td, api) })],
          }))
        })
        rows.push(new TableRow({ children: cells }))
      })
      return [new Table({ rows }), new Paragraph("")]
    }

    // fallback: flatten children
    const out = []
    for(const ch of Array.from(el.childNodes)){
      out.push(...convertBlock(ch, api))
    }
    return out
  }

  function inlineRuns(container, api){
    const { TextRun, ImageRun } = api
    const runs = []

    function walk(node, st){
      st = st || { bold:false, ital:false, under:false }

      if(node.nodeType === Node.TEXT_NODE){
        const text = node.textContent || ''
        if(text) runs.push(new TextRun({ text, bold: st.bold, italics: st.ital, underline: st.under ? {} : undefined }))
        return
      }
      if(node.nodeType !== Node.ELEMENT_NODE) return

      const el = node
      const tag = el.tagName.toLowerCase()

      const next = { ...st }
      if(tag === 'b' || tag === 'strong') next.bold = true
      if(tag === 'i' || tag === 'em') next.ital = true
      if(tag === 'u') next.under = true

      if(tag === 'br'){ runs.push(new TextRun({ text: "\n" })); return }

      if(tag === 'img'){
        const src = el.getAttribute('src') || ''
        if(src.startsWith('data:image/')){
          try{
            const bin = dataUrlToUint8(src)
            runs.push(new ImageRun({ data: bin, transformation: { width: 520, height: 300 } }))
          }catch(e){}
        }
        return
      }

      for(const ch of Array.from(el.childNodes)){
        walk(ch, next)
      }
    }

    walk(container, null)
    return runs.length ? runs : [new TextRun("")]
  }

  function dataUrlToUint8(dataUrl){
    const base64 = dataUrl.split(',')[1] || ''
    const binStr = atob(base64)
    const len = binStr.length
    const bytes = new Uint8Array(len)
    for(let i=0;i<len;i++) bytes[i] = binStr.charCodeAt(i)
    return bytes
  }

  // =========================================================
  // PDF export + Print
  // =========================================================
  $('btnPdf').onclick = async () => {
    try{
      if(typeof html2pdf === 'undefined'){
        showModal('PDF export not available', 'Missing ./js/html2pdf.bundle.min.js')
        return
      }
      setStatus('Generating PDF')

      // pdf: use doc container (pages)
      const opt = {
        margin: 0,
        filename: `${safeName()}.pdf`,
        image: { type: 'jpeg', quality: 0.98 },
        html2canvas: { scale: 2, backgroundColor: '#ffffff' },
        jsPDF: { unit: 'pt', format: 'a4', orientation: 'portrait' }
      }

      await html2pdf().set(opt).from(docEl).save()
      setStatus('PDF downloaded')
    }catch(e){
      showModal('PDF failed', 'Unable to generate PDF in this browser. Try latest Chrome/Edge/Firefox')
    }
  }

  $('btnPrint').onclick = () => {
    window.print()
  }

  // =========================================================
  // Autosave
  // =========================================================
  function saveLocal(){
    try{
      const data = {
        name: currentName,
        readOnly,
        activePageIndex,
        pages: getAllBodies().map(b => b.innerHTML)
      }
      localStorage.setItem(LS_KEY, JSON.stringify(data))
    }catch(e){}
  }

  function loadLocal(){
    try{
      const s = localStorage.getItem(LS_KEY)
      if(!s) return false
      const data = JSON.parse(s)
      if(!data || !Array.isArray(data.pages)) return false

      currentName = data.name || currentName
      clearAllPages()
      // build pages
      data.pages.forEach((html, idx) => {
        if(idx > 0) addPage(idx - 1)
        const b = getBody(idx)
        if(b) b.innerHTML = html || '<p><br></p>'
      })
      activePageIndex = Math.max(0, Math.min(data.activePageIndex || 0, getPages().length - 1))
      setReadOnly(!!data.readOnly)
      renumberPages()
      renderSidebar()
      reflowFrom(0)
      return true
    }catch(e){
      return false
    }
  }

  // =========================================================
  // Wiring: page events
  // =========================================================
  function wirePageEvents(){
    getAllBodies().forEach((b, idx) => {
      // avoid double-binding
      if(b._wired) return
      b._wired = true

      b.addEventListener('focusin', () => {
        activePageIndex = idx
        renderSidebar()
      })

      b.addEventListener('input', () => {
        if(readOnly) return
        activePageIndex = idx
        reflowFrom(idx)
      })

      b.addEventListener('keydown', (e) => {
        // Ctrl+S save docx
        if((e.ctrlKey || e.metaKey) && e.key.toLowerCase() === 's'){
          e.preventDefault()
          $('btnSaveDocx').click()
        }
      })
    })
  }

  // Helper to extract full HTML with page breaks (for DOC)
  function getFullHtml(){
    const bodies = getAllBodies()
    return bodies.map((b, i) => {
      const html = b.innerHTML || ''
      const wrap = `<div class="pageBreak">${html}</div>`
      return (i < bodies.length - 1) ? wrap : html
    }).join('')
  }

  // Filename helper
  function safeName(){
    const base = (currentName || 'freehtml.app_document')
      .replace(/[^\w\- ]+/g,'')
      .trim()
      .slice(0, 60) || 'freehtml.app_document'

    const d = new Date()
    const pad = (n) => String(n).padStart(2,'0')
    return `${base}_${d.getFullYear()}-${pad(d.getMonth()+1)}-${pad(d.getDate())}_${pad(d.getHours())}${pad(d.getMinutes())}`
  }

  function downloadBlob(blob, filename){
    const url = URL.createObjectURL(blob)
    const a = document.createElement('a')
    a.href = url
    a.download = filename
    document.body.appendChild(a)
    a.click()
    a.remove()
    URL.revokeObjectURL(url)
  }

  // =========================================================
  // Init
  // =========================================================
  function init(){
    ensureAtLeastOnePage()
    wirePageEvents()

    if(!loadLocal()){
      setReadOnly(false)
      setDefaultDoc()
      renderSidebar()
      saveLocal()
    }

    // Re-wire after load as pages may be recreated
    wirePageEvents()
    focusPage(activePageIndex)

    // periodic autosave
    setInterval(saveLocal, 15000)
  }

  init()
})()
